/************************************************************************************//**
 * @file		usbIoCard.cpp
 * @brief		Source file for usbIoCard Class
 * @date		2013-11-27
 * @author		Research & Engineering Center Sp. z o.o.
 * @copyright	Hatteland Display AS
 ***************************************************************************************/

#include "stdafx.h"
#include "usbIoCard.h"
#include "ftd2xx.h"

using namespace dio;
using namespace std;

/** @brief FT232H MPSSE commands codes */
enum commandCode : BYTE
{
	SET_DTA_BITS_LO =		0x80, /**< Set data bits low byte (ADBUS 0-7) */
	SET_DTA_BITS_HI =		0x82, /**< Set data bits high byte (ACBUS 0-7) */
	READ_DTA_BITS_LO =		0x81, /**< Read data bits low byte (ADBUS 0-7) */
	READ_DTA_BITS_HI =		0x83, /**< Read data bits high byte (ACBUS 0-7) */
	UNSUPPORTED_COMMAND_1 =	0xAA, /**< Unsupported command */
	UNSUPPORTED_COMMAND_2 =	0xAB, /**< Unsupported command */
	BAD_COMMAND =			0xFA, /**< Bad command code (returned when unsupported command was sent ) */
};

static const unsigned char input = 0;  /**< Bit value for input pin configuration */
static const unsigned char output = 1;  /**< Bit value for output pin configuration */

static const unsigned char loState = 0;  /**< Bit value for low state of pin */
static const unsigned char hiState = 1;  /**< Bit value for high state of pin */

static const unsigned char adbus0 = 0; /**< Bit number for ADBUS0 pin */
static const unsigned char adbus1 = 1; /**< Bit number for ADBUS1 pin */
static const unsigned char adbus2 = 2; /**< Bit number for ADBUS2 pin */
static const unsigned char adbus3 = 3; /**< Bit number for ADBUS3 pin */
static const unsigned char adbus4 = 4; /**< Bit number for ADBUS4 pin */
static const unsigned char adbus5 = 5; /**< Bit number for ADBUS5 pin */
static const unsigned char adbus6 = 6; /**< Bit number for ADBUS6 pin */
static const unsigned char adbus7 = 7; /**< Bit number for ADBUS7 pin */

static const unsigned char acbus0 = 0; /**< Bit number for ACBUS0 pin */
static const unsigned char acbus1 = 1; /**< Bit number for ACBUS1 pin */
static const unsigned char acbus2 = 2; /**< Bit number for ACBUS2 pin */
static const unsigned char acbus3 = 3; /**< Bit number for ACBUS3 pin */
static const unsigned char acbus4 = 4; /**< Bit number for ACBUS4 pin */
static const unsigned char acbus5 = 5; /**< Bit number for ACBUS5 pin */
static const unsigned char acbus6 = 6; /**< Bit number for ACBUS6 pin */
static const unsigned char acbus7 = 7; /**< Bit number for ACBUS7 pin */

static const unsigned char adbus0_in = (input << adbus0); /**< Value for ADBUS0 input configuration */
static const unsigned char adbus0_out = (output << adbus0); /**< Value for ADBUS0 output configuration */
static const unsigned char adbus1_in = (input << adbus1); /**< Value for ADBUS1 input configuration */
static const unsigned char adbus1_out = (output << adbus1); /**< Value for ADBUS1 output configuration */
static const unsigned char adbus2_in = (input << adbus2); /**< Value for ADBUS2 input configuration */
static const unsigned char adbus2_out = (output << adbus2); /**< Value for ADBUS2 output configuration */
static const unsigned char adbus3_in = (input << adbus3); /**< Value for ADBUS3 input configuration */
static const unsigned char adbus3_out = (output << adbus3); /**< Value for ADBUS3 output configuration */
static const unsigned char adbus4_in = (input << adbus4); /**< Value for ADBUS4 input configuration */
static const unsigned char adbus4_out = (output << adbus4); /**< Value for ADBUS4 output configuration */
static const unsigned char adbus5_in = (input << adbus5); /**< Value for ADBUS5 input configuration */
static const unsigned char adbus5_out = (output << adbus5); /**< Value for ADBUS5 output configuration */
static const unsigned char adbus6_in = (input << adbus6); /**< Value for ADBUS6 input configuration */
static const unsigned char adbus6_out = (output << adbus6); /**< Value for ADBUS6 output configuration */
static const unsigned char adbus7_in = (input << adbus7); /**< Value for ADBUS7 input configuration */
static const unsigned char adbus7_out = (output << adbus7); /**< Value for ADBUS7 output configuration */

static const unsigned char acbus0_in = (input << acbus0); /**< Value for ACBUS0 input configuration */
static const unsigned char acbus0_out = (output << acbus0); /**< Value for ACBUS0 output configuration */
static const unsigned char acbus1_in = (input << acbus1); /**< Value for ACBUS1 input configuration */
static const unsigned char acbus1_out = (output << acbus1); /**< Value for ACBUS1 output configuration */
static const unsigned char acbus2_in = (input << acbus2); /**< Value for ACBUS2 input configuration */
static const unsigned char acbus2_out = (output << acbus2); /**< Value for ACBUS2 output configuration */
static const unsigned char acbus3_in = (input << acbus3); /**< Value for ACBUS3 input configuration */
static const unsigned char acbus3_out = (output << acbus3); /**< Value for ACBUS3 output configuration */
static const unsigned char acbus4_in = (input << acbus4); /**< Value for ACBUS4 input configuration */
static const unsigned char acbus4_out = (output << acbus4); /**< Value for ACBUS4 output configuration */
static const unsigned char acbus5_in = (input << acbus5); /**< Value for ACBUS5 input configuration */
static const unsigned char acbus5_out = (output << acbus5); /**< Value for ACBUS5 output configuration */
static const unsigned char acbus6_in = (input << acbus6); /**< Value for ACBUS6 input configuration */
static const unsigned char acbus6_out = (output << acbus6); /**< Value for ACBUS6 output configuration */
static const unsigned char acbus7_in = (input << acbus7); /**< Value for ACBUS7 input configuration */
static const unsigned char acbus7_out = (output << acbus7); /**< Value for ACBUS7 output configuration */

static const unsigned char acbus0_lo = (loState << acbus0); /**< Value for ACBUS0 low state */
static const unsigned char acbus0_hi = (hiState << acbus0); /**< Value for ACBUS0 high state */
static const unsigned char acbus1_lo = (loState << acbus1); /**< Value for ACBUS1 low state */
static const unsigned char acbus1_hi = (hiState << acbus1); /**< Value for ACBUS1 high state */
static const unsigned char acbus2_lo = (loState << acbus2); /**< Value for ACBUS2 low state */
static const unsigned char acbus2_hi = (hiState << acbus2); /**< Value for ACBUS2 high state */
static const unsigned char acbus3_lo = (loState << acbus3); /**< Value for ACBUS3 low state */
static const unsigned char acbus3_hi = (hiState << acbus3); /**< Value for ACBUS3 high state */

static const unsigned char in0_cfg = adbus4_in; /**< Value for IN0 configuration */
static const unsigned char in1_cfg = adbus5_in; /**< Value for IN1 configuration */
static const unsigned char in2_cfg = adbus6_in; /**< Value for IN2 configuration */
static const unsigned char in3_cfg = adbus7_in; /**< Value for IN3 configuration */

static const unsigned char gpo0_cfg = acbus0_out; /**< Value for GPO0 configuration */
static const unsigned char gpo1_cfg = acbus1_out; /**< Value for GPO1 configuration */
static const unsigned char gpo2_cfg = acbus2_out; /**< Value for GPO2 configuration */
static const unsigned char gpo3_cfg = acbus3_out; /**< Value for GPO3 configuration */

static const unsigned char gpi0_cfg = acbus4_in; /**< Value for GPI0 configuration */
static const unsigned char gpi1_cfg = acbus5_in; /**< Value for GPI1 configuration */
static const unsigned char gpi2_cfg = acbus6_in; /**< Value for GPI2 configuration */
static const unsigned char gpi3_cfg = acbus7_in; /**< Value for GPI3 configuration */

static const unsigned char adbus_cfg = in3_cfg | in2_cfg | in1_cfg | in0_cfg | adbus3_in | adbus2_in | adbus1_in | adbus0_in; /**< ADBUS port configuration */
static const unsigned char acbus_cfg = gpi3_cfg | gpi2_cfg | gpi1_cfg | gpi0_cfg | gpo3_cfg | gpo2_cfg | gpo1_cfg | gpo0_cfg; /**< ACBUS port configuration */

static const unsigned char gpo0_lo = acbus0_hi; /**< Value for GPO0 low state */
static const unsigned char gpo0_hi = acbus0_hi; /**< Value for GPO0 high state */
static const unsigned char gpo1_lo = acbus1_hi; /**< Value for GPO1 low state */
static const unsigned char gpo1_hi = acbus1_hi; /**< Value for GPO1 high state */
static const unsigned char gpo2_lo = acbus2_hi; /**< Value for GPO2 low state */
static const unsigned char gpo2_hi = acbus2_hi; /**< Value for GPO2 high state */
static const unsigned char gpo3_lo = acbus3_hi; /**< Value for GPO3 low state */
static const unsigned char gpo3_hi = acbus3_hi; /**< Value for GPO3 high state */

static const unsigned char gpi0_mask = acbus4_out; /**< Mask for GPI0 state */
static const unsigned char gpi1_mask = acbus5_out; /**< Mask for GPI1 state */
static const unsigned char gpi2_mask = acbus6_out; /**< Mask for GPI2 state */
static const unsigned char gpi3_mask = acbus7_out; /**< Mask for GPI3 state */

static const unsigned char in0_mask = adbus4_out; /**< Mask for IN0 state */
static const unsigned char in1_mask = adbus5_out; /**< Mask for IN1 state */
static const unsigned char in2_mask = adbus6_out; /**< Mask for IN2 state */
static const unsigned char in3_mask = adbus7_out; /**< Mask for IN3 state */

/************************************************************************************//**
 * @brief Class constructor
 ***************************************************************************************/
usbIoCard_c::usbIoCard_c()
{
	ftHandle = NULL;
}
/************************************************************************************//**
 * @brief Class destructor
 ***************************************************************************************/
usbIoCard_c::~usbIoCard_c()
{
	close();
}
/************************************************************************************//**
 * @brief Function opens the USB IO card
 * @return NO_ERR USB IO card opened
 * @return NA_ERR USB IO card is not connected
 * @return OPEN_ERR USB IO card cannot be opened
 ***************************************************************************************/
errorCode_e usbIoCard_c::open()
{
	static const char* deviceName = "Hatteland USB->Digital IO Card";
	
	DWORD cardNum;
	if (FT_OK == FT_CreateDeviceInfoList(&cardNum))
	{
		if (0 != cardNum)
		{
			try
			{
				FT_DEVICE_LIST_INFO_NODE* cardInfo = new FT_DEVICE_LIST_INFO_NODE [cardNum];
				if (FT_OK == FT_GetDeviceInfoList(cardInfo, &cardNum))
				{
					static const int noUsbIoCard = 0xFFFFFFFF;
					int deviceNumber = noUsbIoCard;
					for (unsigned int i = 0; i < cardNum; i++)
					{
						if (0 == strcmp(deviceName, cardInfo[i].Description))
						{
							deviceNumber = i;
							break;
						}
					}
					delete[] cardInfo;
					cardInfo = NULL;
					if (noUsbIoCard != deviceNumber)
					{
						if (FT_OK != FT_Open(deviceNumber, &ftHandle))
						{
							return OPEN_ERR;
						}
					}
					else
					{
						return NA_ERR;
					}
				}
				else
				{
					delete [] cardInfo;
					cardInfo = NULL;
					return OPEN_ERR;
				}
			}
			catch (bad_alloc&)
			{
				return OPEN_ERR;
			};
		}
		else
		{
			return NA_ERR;
		}
	}
	else
	{
		return OPEN_ERR;
	}
	return NO_ERR;
}
/************************************************************************************//**
 * @brief Function closes the USB IO card
 * @return NO_ERR USB IO card closed
 * @return CLOSE_ERR USB IO card cannot be closed
 ***************************************************************************************/
errorCode_e usbIoCard_c::close()
{
	if (NULL != ftHandle)
	{
		if (FT_OK != FT_Close(ftHandle))
		{
			return CLOSE_ERR;
		}
		ftHandle = NULL;
	}
	return NO_ERR;
}
/************************************************************************************//**
 * @brief Function configures USB IO card
 * @return NO_ERR USB IO card configured
 * @return CONFIG_ERR USB IO card has not been configured
 ***************************************************************************************/
errorCode_e usbIoCard_c::configure()
{
	errorCode_e err = verifyMpsse();
	if (MPSSE_ERR == err)
	{
		FT_STATUS ftStatus = FT_OK;
		// Reset the peripheral side of FTDI port
		ftStatus |= FT_ResetDevice(ftHandle); 
		// purge FTDI buffers
		ftStatus |= FT_Purge(ftHandle, (FT_PURGE_TX | FT_PURGE_RX));
		// Configure the maximum USB transfer sizes
		static const ULONG inTransferSize = 64;
		static const ULONG outTransferSize = 64;
		ftStatus |= FT_SetUSBParameters(ftHandle, inTransferSize, outTransferSize);
		// Configure the event and error characters (Most applications disable any event and error characters)
		static const UCHAR charNone = 0x00;
		static const UCHAR eventDisabled = 0x00;
		ftStatus |= FT_SetChars(ftHandle, charNone, eventDisabled, charNone, eventDisabled);
		// Configure the read and write timeouts in miliseconds
		static const ULONG timeoutDisabled = 0;
		static const ULONG readTimeout = timeoutDisabled;
		static const ULONG writeTimeout = timeoutDisabled;
		ftStatus |= FT_SetTimeouts(ftHandle, readTimeout, writeTimeout); // It will not use timeouts
		// Configure the amount of time to wait before sending an incomplete USB packet from the peripheral back to the host
		// For applications that require quick responses from peripheral latency timer should be set to a lower value
		static const UCHAR latencyTimerVal = 2;
		ftStatus |= FT_SetLatencyTimer(ftHandle, latencyTimerVal);
		// Configure flow control for RTS/CTS to ensure that the driver will not issue IN requests if the buffer is unable to accept data
		ftStatus |= FT_SetFlowControl(ftHandle, FT_FLOW_RTS_CTS, charNone, charNone);
		// Perform a general reset on the MPSSE, not the port itself
		static const UCHAR bitModeMask = 0x00;
		static const UCHAR mpsseReset = 0x00;
		ftStatus |= FT_SetBitMode(ftHandle, bitModeMask, mpsseReset);
		// Enable the MPSSE controller
		static const UCHAR mpsseMode = 0x02;
		ftStatus |= FT_SetBitMode(ftHandle, bitModeMask, mpsseMode);
		if (FT_OK != ftStatus)
		{
			return CONFIG_ERR;
		}
		Sleep(50); // Wait for all the USB stuff to complete and work (delay due to AN_135)
		err = verifyMpsse();
		if (MPSSE_ERR == err)
		{
			return CONFIG_ERR;
		}
		else if (WRITE_ERR == err)
		{
			// Reset MPSSE mode to ensure handling of another configuration procedure (in case of WRITE_ERR)
			if (FT_OK != FT_SetBitMode(ftHandle, bitModeMask, mpsseReset))
			{
				return CONFIG_ERR;
			}
			return CONFIG_ERR;
		}
		if (NO_ERR != configIoPinDir())
		{
			// Reset MPSSE mode to ensure handling of another configuration procedure (in case of WRITE_ERR)
			if (FT_OK != FT_SetBitMode(ftHandle, bitModeMask, mpsseReset))
			{
				return CONFIG_ERR;
			}
			return CONFIG_ERR;
		}
	}
	else if (WRITE_ERR == err)
	{
		return CONFIG_ERR;
	}
	return NO_ERR;
}
/************************************************************************************//**
 * @brief Function sets output state
 * @param[in] output Name of output
 * @param[in] state New state of output
 * @return NO_ERR Output state changed
 * @return OUTPUT_NAME_ERR Unsupported output name
 * @return OUTPUT_STATE_ERR Unsupported output state
 * @return OUTPUT_ERR State of an output has not been changed
 ***************************************************************************************/
errorCode_e usbIoCard_c::setOutputState(outputName_e output, outputStateName_e state)
{
	if (GPIO_NA_NAME <= output)
	{
		return OUTPUT_NAME_ERR;
	}
	if (OUT_NA_STATE <= state)
	{
		return OUTPUT_STATE_ERR;
	}
	// Write command for read the current state of port
	static const unsigned char bufferSize = 3;
	unsigned char buffer[bufferSize];
	DWORD bytesToTransfer = 0;
	buffer[bytesToTransfer++] = READ_DTA_BITS_HI;
	if (NO_ERR != writeData(buffer, bytesToTransfer))
	{
		return OUTPUT_ERR;
	}
	// Read the current state of port
	bytesToTransfer = 1;
	if (NO_ERR != readData(buffer, bytesToTransfer))
	{
		return OUTPUT_ERR;
	}
	unsigned char portState = buffer[0];
	unsigned char outState;
	switch (output)
	{
		case GPO0:
			outState = gpo0_hi;
			break;
		case GPO1:
			outState = gpo1_hi;
			break;
		case GPO2:
			outState = gpo2_hi;
			break;
		case GPO3:
			outState = gpo3_hi;
			break;
	}
	if (OUT_HI == state)
	{
		portState &= ~outState;
	}
	else
	{
		portState |= outState;
	}
	// Write command to change state of port
	bytesToTransfer = 0;
	buffer[bytesToTransfer++] = SET_DTA_BITS_HI;
	buffer[bytesToTransfer++] = portState;
	buffer[bytesToTransfer++] = acbus_cfg;
	if (NO_ERR != writeData(buffer, bytesToTransfer))
	{
		return OUTPUT_ERR;
	}
	return NO_ERR;
}
/************************************************************************************//**
 * @brief Function diagnoses output state
 * @param[in] output Name of output
 * @param[out] state State of output returned trough reference
 * @return NO_ERR Output state diagnosed
 * @return OUTPUT_NAME_ERR Unsupported output name
 * @return OUTPUT_ERR State of an output has not been diagnosed
 ***************************************************************************************/
errorCode_e usbIoCard_c::diagOutputState(outputName_e output, outputStateName_e& state)
{
	if (GPIO_NA_NAME <= output)
	{
		return OUTPUT_NAME_ERR;
	}
	// Write command for read the current state of port
	static const unsigned char bufferSize = 1;
	unsigned char buffer[bufferSize];
	DWORD bytesToTransfer = 0;
	buffer[bytesToTransfer++] = READ_DTA_BITS_HI;
	if (NO_ERR != writeData(buffer, bytesToTransfer))
	{
		return OUTPUT_ERR;
	}
	// Read the current state of port
	bytesToTransfer = 1;
	if (NO_ERR != readData(buffer, bytesToTransfer))
	{
		return OUTPUT_ERR;
	}
	unsigned char portState = buffer[0];
	unsigned char inStateMask;
	switch (output)
	{
		case GPO0:
			inStateMask = gpi0_mask;
			break;
		case GPO1:
			inStateMask = gpi1_mask;
			break;
		case GPO2:
			inStateMask = gpi2_mask;
			break;
		case GPO3:
			inStateMask = gpi3_mask;
			break;
	}
	if (0 != (portState & inStateMask))
	{
		state = OUT_LO; // because of inverted logic on GPI inputs
	}
	else
	{
		state = OUT_HI; // because of inverted logic on GPI inputs
	}
	return NO_ERR;
}
/************************************************************************************//**
 * @brief Function reads input state
 * @param[in] input Name of input
 * @param[out] state State of input returned trough reference
 * @return NO_ERR Input state read
 * @return INPUT_NAME_ERR Unsupported input name
 * @return INPUT_ERR State of an input has not been read
 ***************************************************************************************/
errorCode_e usbIoCard_c::readInputState(inputName_e input, inputStateName_e& state)
{
	if (IN_NA_NAME <= input)
	{
		return INPUT_NAME_ERR;
	}
	// Write command for read the current state of port
	static const unsigned char bufferSize = 1;
	unsigned char buffer[bufferSize];
	DWORD bytesToTransfer = 0;
	buffer[bytesToTransfer++] = READ_DTA_BITS_LO;
	if (NO_ERR != writeData(buffer, bytesToTransfer))
	{
		return INPUT_ERR;
	}
	// Read the current state of port
	bytesToTransfer = 1;
	if (NO_ERR != readData(buffer, bytesToTransfer))
	{
		return INPUT_ERR;
	}
	unsigned char portState = buffer[0];
	unsigned char inStateMask;
	switch (input)
	{
		case IN0:
			inStateMask = in0_mask;
			break;
		case IN1:
			inStateMask = in1_mask;
			break;
		case IN2:
			inStateMask = in2_mask;
			break;
		case IN3:
			inStateMask = in3_mask;
			break;
	}
	if (0 != (portState & inStateMask))
	{
		state = IN_LO; // because of inverted logic on IN inputs
	}
	else
	{
		state = IN_HI; // because of inverted logic on IN inputs
	}
	return NO_ERR;
}
/************************************************************************************//**
 * @brief Function verifies MPSSE mode
 * @return NO_ERR USB IO card in MPSSE mode
 * @return MPSSE_ERR USB IO card is not in MPSSE mode
 * @return WRITE_ERR Data cannot be written to USB IO card
 ***************************************************************************************/
errorCode_e usbIoCard_c::verifyMpsse()
{
	static const unsigned char bufferSize = 2;
	unsigned char buffer[bufferSize];

	DWORD bytesToTransfer = 0;
	buffer[bytesToTransfer++] = UNSUPPORTED_COMMAND_1;
	if (NO_ERR != writeData(buffer, bytesToTransfer))
	{
		return WRITE_ERR;
	}
	bytesToTransfer = bufferSize;
	if (NO_ERR != readData(buffer, bytesToTransfer))
	{
		return MPSSE_ERR;
	}
	if ((BAD_COMMAND != buffer[0]) || (UNSUPPORTED_COMMAND_1 != buffer[1]))
	{
		return MPSSE_ERR;
	}
	return NO_ERR;
}
/************************************************************************************//**
 * @brief Function configures direction of USB IO card pins
 * @return NO_ERR Direction of USB IO card pins set
 * @return CONFIG_ERR USB IO card is not in MPSSE mode
 ***************************************************************************************/
errorCode_e usbIoCard_c::configIoPinDir()
{
	static const unsigned char bufferSize = 3;
	static const unsigned char lowState = 0x0F;
	unsigned char buffer[bufferSize];

	DWORD bytesToTransfer = 0;
	buffer[bytesToTransfer++] = SET_DTA_BITS_LO;
	buffer[bytesToTransfer++] = lowState;
	buffer[bytesToTransfer++] = adbus_cfg;
	if (NO_ERR != writeData(buffer, bytesToTransfer))
	{
		return CONFIG_ERR;
	}
	bytesToTransfer = 0;
	buffer[bytesToTransfer++] = SET_DTA_BITS_HI;
	buffer[bytesToTransfer++] = lowState;
	buffer[bytesToTransfer++] = acbus_cfg;
	bytesToTransfer = bufferSize;
	if (NO_ERR != writeData(buffer, bytesToTransfer))
	{
		return CONFIG_ERR;
	}
	return NO_ERR;
}
/************************************************************************************//**
 * @brief Function reads data from USB IO card
 * @param[in] dataBuffer Pointer to a buffer for read data
 * @param[in] bytesToRead Number of bytes to read
 * @return NO_ERR Data written to USB IO card
 * @return READ_ERR Data cannot be written to USB IO card
 * @return NO_DATA_ERR There is no data to read in USB IO card
 ***************************************************************************************/
errorCode_e usbIoCard_c::readData(unsigned char* dataBuffer, unsigned int bytesToRead)
{
	static const unsigned int timeoutValue = 10000;

	if (0 == bytesToRead)
	{
		return NO_ERR;
	}
	if (NULL == dataBuffer)
	{
		return READ_ERR;
	}
	memset(dataBuffer, 0x00, bytesToRead); // clear destination buffer
	unsigned int timeoutCounter = 0;
	DWORD bytesInQueue = 0;
	while ((bytesInQueue < bytesToRead) && (timeoutCounter < timeoutValue))
	{
		if (FT_OK != FT_GetQueueStatus(ftHandle, &bytesInQueue))
		{
			return READ_ERR;
		}
		timeoutCounter++;
	}
	if (timeoutValue <= timeoutCounter)
	{
		return NO_DATA_ERR;
	}
	DWORD bytesRead;
	if (FT_OK != FT_Read(ftHandle, dataBuffer, bytesToRead, &bytesRead))
	{
		return READ_ERR;
	}
	return NO_ERR;
}
/************************************************************************************//**
 * @brief Function writes data to USB IO card
 * @param[in] dataBuffer Pointer to a buffer with data to write
 * @param[in] bytesToWrite Number of bytes in the dataBuffer
 * @return NO_ERR Data written to USB IO card
 * @return WRITE_ERR Data cannot be written to USB IO card
 ***************************************************************************************/
errorCode_e usbIoCard_c::writeData(unsigned char* dataBuffer, unsigned int bytesToWrite)
{
	if (0 == bytesToWrite)
	{
		return NO_ERR;
	}
	if (NULL == dataBuffer)
	{
		return WRITE_ERR;
	}
	DWORD bytesWritten;
	if (FT_OK != FT_Write(ftHandle, dataBuffer, bytesToWrite, &bytesWritten))
	{
		return WRITE_ERR;
	}
	if (bytesToWrite != bytesWritten)
	{
		return WRITE_ERR;
	}
	return NO_ERR;
}
/***************************************************************************************/
